using System;
using System.Collections.Generic;
using System.Text;
using System.Data.SQLite;
using System.Data;
using System.IO;

namespace BookShop.Model
{
    /// <summary>
    /// Data Access Object minta: olyan osztly ami a felsbb rtegek fel az adatok elrst
    /// biztostja. Alapvet adatmanipulcis mveletek megvalstsa: CRUD.
    /// 
    /// C = Create (j adat ltrehozsa)
    /// R = Read (adatok lekrdezse felttel(ek) alapjn)
    /// U = Update (meglv adatok mdostsa)
    /// D = Delete (adatok trlse)
    /// </summary>
    class BookShopDao
    {
        /// <summary>
        /// SQLite specifikus connection stringnk, relatv tmegadssal.
        /// A Webes alkalmazs a windows/system32-ben fut, ezrt ott absz. tvonal kell
        /// </summary>
        private static readonly String connectionString = @"Data Source=d:\alkfejl\kulso\db\bookshop.db;";

        /// <summary>
        /// ebben troljuk az gyfeleket, memriban: nv / gyfl prknt
        /// </summary>
        private Dictionary<String, Customer> customers = new Dictionary<string, Customer>();
        /// <summary>
        /// ebben meg a knyveket
        /// </summary>
        private Dictionary<String, Book> books = new Dictionary<string, Book>();

        /// <summary>
        /// j gyfl hozzadsa.
        /// </summary>
        /// <param name="customer">az gyfl pldny amit eltrolunk</param>
        /// <returns>true, ha sikerlt a hozzads, false, ha mr ltezik azonos nev gyfl</returns>
        public bool AddCustomer(Customer customer)
        {
            if (DoesCustomerExist(customer))
            {
                return false;
            }
            else
            {
                return InsertCustomer(customer);
            }
        }


        /// <summary>
        /// Ellenrzi, hogy egy vsrl esetleg ltezik-e mr.
        /// Az egyszersg kedvrt a vsrl nevt tekintjk 
        /// azonostnak.
        /// </summary>
        /// <returns>true, ha ltezik mr a felhasznl,
        /// egybknt false</returns>
        private bool DoesCustomerExist(Customer customer)
        {
            // true-val kezdnk, mivel ez nagyobb biztonsgot jelent
            // az adatbzis logikra nzve, ha false maradna egy esetleg 
            // hiba miatt, akkor a felhasznl kd megprblhatna mr egy 
            // ltez felhasznlt felvenni
            bool rvExists = true;

            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            using (SQLiteCommand cmd = conn.CreateCommand())
            {
                // elksztjk a parancsot, egy egyszer paramteres lekrdezs
                // a customer nevt felhasznlva
                cmd.CommandText = "SELECT id From Customer WHERE name = @name";

                // regisztrljuk a paramtert a nv szmra
                SQLiteParameter nameParam = 
                    cmd.Parameters.Add("@name", DbType.String);
                nameParam.Value = customer.Name;

                conn.Open();
                object result = cmd.ExecuteScalar();

                rvExists = result != null;
            }

            return rvExists;
        }


        /// <summary>
        /// Egy j vsrlt vesz fel az adatbzisba.
        /// </summary>
        /// <param name="customer"></param>
        /// <returns>true, ha a mvelet sikeres, egybknt false</returns>
        private bool InsertCustomer(Customer customer)
        {
            bool rvSuccess = false;

            // kapcsolati objektumok ltrehozsa
            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            using (SQLiteCommand cmd = conn.CreateCommand())
            {
                // literlokkal vgezhetnk mveleteket, mert a C# fordt vgez constant folding-ot,
                // azaz a fordtsi idben elvgezhet mveleteket elvgzi.
                cmd.CommandText = "INSERT INTO Customer (name, age, female, rented, student, grantee, qualif) " +
                                  "VALUES (@name, @age, @female, @rented, @student, @grantee, @qualif)";


                //regisztrljuk s egyttal be is lltjuk a paramtereket
                // A Convert osztly a boolean -> int32 talaktst
                // ignyeinknek megfelelen vgzi el:
                // true -> 1, false -> 0
                cmd.Parameters.Add("@name", DbType.String).Value = customer.Name;
                cmd.Parameters.Add("@age", DbType.Int32).Value = customer.Age;
                cmd.Parameters.Add("@female", DbType.Int32).Value = Convert.ToInt32(customer.Female);
                cmd.Parameters.Add("@rented", DbType.Int32).Value = Convert.ToInt32(customer.Rented);
                cmd.Parameters.Add("@student", DbType.Int32).Value = Convert.ToInt32(customer.Student);
                cmd.Parameters.Add("@grantee", DbType.Int32).Value = Convert.ToInt32(customer.Grantee);
                cmd.Parameters.Add("@qualif", DbType.String).Value = customer.Qualification;

                conn.Open();
                int rowCount = cmd.ExecuteNonQuery();

                rvSuccess = rowCount == 1;
            }

            return rvSuccess;
        }


        /// <summary>
        /// gyfelek listjnak lekrse.
        /// </summary>
        /// <returns>gyfelek kollekcija</returns>
        public IEnumerable<Customer> GetAllCustomers()
        {
            customers.Clear();

            // kapcsolati objektumok ltrehozsa
            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            using (SQLiteCommand cmd = conn.CreateCommand())
            {
                // a teljes tbla tartalmt le kvnjuk krdezni
                // lehetne gy is, de nem tmogatott (kivtelt kapunk)
                //cmd.CommandType = CommandType.TableDirect;
                //cmd.CommandText = "Customer";
                cmd.CommandText = "SELECT * FROM Customer";

                conn.Open();
                // vgrehajtjuk a lekrdezst, visszakapva az eredmnyhalmazt
                // DataReader is IDisposable, ezrt using-ot hasznlunk
                using (SQLiteDataReader reader = cmd.ExecuteReader())
                {
                    // amg van j eredmnysor, addih olvasunk
                    while (reader.Read())
                    {
                        // minden egyes sorbon ltehozunk egy objektumot
                        Customer c = new Customer();
                        c.Id = reader.GetInt32(reader.GetOrdinal("id"));
                        c.Name = reader.GetString(reader.GetOrdinal("name"));
                        c.Age = reader.GetInt32(reader.GetOrdinal("age"));
                        // !!! a GetBoolean automatikusan konvertlta az rtket
                        // az adatbzisban int-knt troljuk
                        c.Female = reader.GetBoolean(reader.GetOrdinal("female"));
                        c.Rented = reader.GetBoolean(reader.GetOrdinal("rented"));
                        c.Student = reader.GetBoolean(reader.GetOrdinal("student"));
                        c.Grantee = reader.GetBoolean(reader.GetOrdinal("grantee"));
                        c.Qualification = reader.GetString(reader.GetOrdinal("qualif"));
                        customers.Add(c.Name, c);
                    }
                }
            }

            return customers.Values;
        }

        /// <summary>
        /// Megnzi hogy az adott knyv szerepel-e mr az adatbzisban
        /// (szerz, cm s kiadsi v alapjn). Ha mr ltezik visszaadja
        /// az Id-jt, egybknt meg null rtket.
        /// 
        /// Az int? tpus megegyezik a sima int tpussal de lehet neki null
        /// rtket is adni. Ez remekl hasznlhat arra hogy jelezznk egy
        /// nemltez rtket. (Hasonlan mkdik brmely msik rtk alap
        /// tpusnl, pl. bool? egy 3-llapot rtk, ami lehet true, false
        /// vagy null.)
        /// </summary>
        /// <param name="book">egy knyv pldny</param>
        /// <returns>a knyv id-je, vagy null ha nincs ilyen az adatbzisban</returns>
        public int? GetBookId(Book book)
        {
            int? id = null;

            // A using kulcssz a nem menedzselt erforrsok kezelst segti
            // a blokkbl kilpve garantlja hogy a paramternek megadott erforrs
            // (itt a 'conn') le lesz zrva (conn.Close();) akr sikeresen lefutottak
            // a mveletek, akr hiba (exception) miatt lp ki a blokkbl.
            //
            // Nem menedzselt erforrsok .Net alatt (s Javaban is!):
            // adatbziskapcsolat, rs/olvass, fjlelrs, hlzati mveletek...
            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            {
                // ltrehozunk egy parancs objektumot, ami SQL parancsok vgrehajtsra
                // lesz alkalmas, ezt mindig egy adott connection-hz ktjk
                SQLiteCommand command = conn.CreateCommand();

                // CommandText property tartalmazza magt a nyers SQL parancsot.
                // Itt most paramteres SQL parancsot adunk, a paramtereket nvvel elltva
                // definilhatjuk .Net-ben.
                command.CommandText = "select id from Book where author=@author and title=@title and year=@year";

                // rtket adunk a paramtereknek. Ehhez elbb lrehozunk egy j paramtert
                // (Add) s megadjuk a nevt + SQL szerinti tpust. A nv azonos az elbb definilt
                // SQL parancsban rt nevekkel. Ezutn a keletkezett (SQLite)Parameter objektum
                // Value property-nek rtkl adhatjuk a paramter rtkt.
                command.Parameters.Add("author", System.Data.DbType.String).Value = book.Author;
                command.Parameters.Add("title", System.Data.DbType.String).Value = book.Title;
                command.Parameters.Add("year", System.Data.DbType.Int32).Value = book.Year;

                // Ellenttben Java-val a connection ltrehozsa nem jelent automatikus csatlakozst
                // az adatbzishoz. A tnyleges mveletvgzs eltt meg kell nyitni a kapcsolatot.
                // Megj.: a vgn le kne zrni is, de ezt a 'using' blokk hasznlata elintzi.
                conn.Open();

                // A parancsot lefuttatjuk. Az ExecuteScalar() metdust akkor hasznlhatjuk, 
                // ha csak egyetlen eredmnyt vrunk. Itt csak egyetlen 'id' mez rtkt
                // krtk le. Arrl persze mr neknk kell gondoskodni, hogy azonos cm+szerz+v
                // adattal ne lehessen kt klnbz knyv az adatbzisban. A metdus object-et
                // ad vissza, amit type castolni kell a vrt eredmnytpusra. Ha nincs eredmny
                // akkor null rtket kapunk.
                object result =  command.ExecuteScalar();
                if (result != null)
                {
                    id = Convert.ToInt32(result);
                }
            }

            return id;
        }

        /// <summary>
        /// Egy knyv hozzadsa az adatbzishoz. Ha mr ltezik a knyv, akkor a meglv kszletet
        /// nveli csak, egybknt hozzad egy jat az Book tblhoz.
        /// </summary>
        /// <param name="book"></param>
        /// <returns></returns>
        public bool AddOrUpdateBook(Book book)
        {
            int? id = GetBookId(book);

            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            {
                SQLiteCommand command = conn.CreateCommand();

                if (id != null)
                {
                    // a knyv mr ltezik update sql parancsot kell hasznlni
                    command.CommandText = "update Book set pieces=pieces+@pieces where id=@id";
                    command.Parameters.Add("id", System.Data.DbType.Int32).Value = id;
                    command.Parameters.Add("pieces", System.Data.DbType.Int32).Value = book.Piece;
                }
                else
                {
                    // a knyv mg nincs az adatbzisban, insert sql parancs kell
                    command.CommandText = "insert into Book (author, title, year, category, price, pieces, ancient) "
                        + " values (@author, @title, @year, @category, @price, @pieces, @ancient)";
                    command.Parameters.Add("author", System.Data.DbType.String).Value = book.Author;
                    command.Parameters.Add("title", System.Data.DbType.String).Value = book.Title;
                    command.Parameters.Add("year", System.Data.DbType.Int32).Value = book.Year;
                    command.Parameters.Add("category", System.Data.DbType.String).Value = book.Category;
                    command.Parameters.Add("price", System.Data.DbType.Int32).Value = book.Price;
                    command.Parameters.Add("pieces", System.Data.DbType.Int32).Value = book.Piece;
                    command.Parameters.Add("ancient", System.Data.DbType.Int32).Value = book.Ancient ? 1 : 0;
                }

                conn.Open();

                // az ExecuteNonQuery() metdus vgrehajtja a parancsot, olyan parancsok esetn
                // hasznljuk ahol nem vrunk vissza eredmnyt (nem Select). A visszatrsi rtke
                // a parancs ltal rintett sorok szma. Ha ez itt nem 1 akkor valami gond van, mivel
                // mi pont egyetlen knyvet szeretnnk beszrni/mdostani.
                if (command.ExecuteNonQuery() != 1)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// A kszleten lv knyvek lekrse. Csak azokat adja vissza ahol a kszet (Piece)
        /// legalbb 1.
        /// </summary>
        /// <returns>knyvek listja</returns>
        public IEnumerable<Book> GetAllBooks()
        {
            // trljk a memriban lv rtkeket
            books.Clear();

            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            {
                SQLiteCommand command = conn.CreateCommand();
                command.CommandText = "select * from Book";

                conn.Open();

                // ExecuteReader()-t hasznlunk ha Select parancsot kldnk s a teljes
                // eredmnyre (tblzat) kvncsiak vagyunk. Az eredmny mindig egy
                // valamilyen DataReader objektum.
                SQLiteDataReader reader = command.ExecuteReader();

                while (reader.Read())
                {
                    Book book = new Book();
                    book.Author = reader["author"].ToString();
                    book.Title = reader["title"].ToString();
                    book.Year = Convert.ToInt32(reader["year"]);
                    book.Category = reader["category"].ToString();
                    book.Price = Convert.ToInt32(reader["price"]);
                    book.Piece = Convert.ToInt32(reader["pieces"]);
                    book.Ancient = (Convert.ToInt32(reader["ancient"]) != 0);
                    book.Id = Convert.ToInt32(reader["id"]);

                    books.Add(book.Title, book);
                }
            }

            return books.Values;
        }


        public bool SellBook(Book book, Customer customer)
        {
            bool rvSuccess = false;

            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            using (SQLiteCommand cmd = conn.CreateCommand())
            {
                cmd.CommandText = "INSERT INTO SoldBookInstances (bookid, customerid, sellDate) " +
                                  "VALUES (@bookid, @customerid, @sellDate)";

                cmd.Parameters.Add("@bookid", DbType.Int32).Value = book.Id;
                cmd.Parameters.Add("@customerid", DbType.Int32).Value = customer.Id;
                cmd.Parameters.Add("@sellDate", DbType.Date).Value = DateTime.Now;

                conn.Open();
                int rowCount = cmd.ExecuteNonQuery();

                rvSuccess = rowCount == 1;
            }

            return rvSuccess;
        }


        public IEnumerable<SoldBook> GetSoldBooks()
        {
            List<SoldBook> soldbooks = new List<SoldBook>();

            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            using (SQLiteCommand cmd = conn.CreateCommand())
            {
                cmd.CommandText = "SELECT * FROM SoldBookInstances";

                conn.Open();
                using (SQLiteDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        SoldBook sb = new SoldBook();
                        sb.SellDate = reader.GetDateTime(reader.GetOrdinal("sellDate"));
                        sb.Customer = GetCustomer(reader.GetInt32(reader.GetOrdinal("customerid")));
                        sb.Book = GetBook(reader.GetInt32(reader.GetOrdinal("bookid")));
                        soldbooks.Add(sb);
                    }
                }
            }

            return soldbooks;
        }


        public Customer GetCustomer(int id)
        {
            Customer c = null;
            
            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            using (SQLiteCommand cmd = conn.CreateCommand())
            {
                cmd.CommandText = "SELECT * FROM Customer WHERE id = @id";

                cmd.Parameters.Add("@id", DbType.Int32).Value = id;

                conn.Open();
                using (SQLiteDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        c = new Customer();
                        c.Id = reader.GetInt32(reader.GetOrdinal("id"));
                        c.Name = reader.GetString(reader.GetOrdinal("name"));
                        c.Age = reader.GetInt32(reader.GetOrdinal("age"));
                        c.Female = reader.GetBoolean(reader.GetOrdinal("female"));
                        c.Rented = reader.GetBoolean(reader.GetOrdinal("rented"));
                        c.Student = reader.GetBoolean(reader.GetOrdinal("student"));
                        c.Grantee = reader.GetBoolean(reader.GetOrdinal("grantee"));
                        c.Qualification = reader.GetString(reader.GetOrdinal("qualif"));
                    }
                }
            }

            return c;
        }

        public Book GetBook(int id)
        {
            Book b = null;

            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            using (SQLiteCommand cmd = conn.CreateCommand())
            {
                cmd.CommandText = "SELECT * FROM Book WHERE id = @id";

                cmd.Parameters.Add("@id", DbType.Int32).Value = id;

                conn.Open();
                using (SQLiteDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        b = new Book();
                        b.Author = reader["author"].ToString();
                        b.Title = reader["title"].ToString();
                        b.Year = Convert.ToInt32(reader["year"]);
                        b.Category = reader["category"].ToString();
                        b.Price = Convert.ToInt32(reader["price"]);
                        b.Piece = Convert.ToInt32(reader["pieces"]);
                        b.Ancient = (Convert.ToInt32(reader["ancient"]) != 0);
                        b.Id = Convert.ToInt32(reader["id"]);
                    }
                }
            }

            return b;
        }

        public List<string> ListQualifications()
        {
            using (SQLiteConnection conn = new SQLiteConnection(connectionString))
            {
                SQLiteCommand command = conn.CreateCommand();
                command.CommandText = "select * from qualifications";

                conn.Open();

                SQLiteDataReader result = command.ExecuteReader();

                List<string> qualifications = new List<string>();
                while (result.Read())
                {
                    qualifications.Add((string)result["qualification"]);
                }
                return qualifications;
            }
        }
    }
}
