import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.*;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.hadoop.mapreduce.lib.input.FileInputFormat;
import org.apache.hadoop.mapreduce.lib.input.NLineInputFormat;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;

import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.StringTokenizer;

public class CurrencyAvg {

    public static class TokenizerMapper
            extends Mapper<Object, Text, Text, DoubleWritable> {

        private final static DoubleWritable currencyPrice = new DoubleWritable();
        private final Text country = new Text();

        public void map(Object key, Text value, Context context
        ) throws IOException, InterruptedException {
            StringTokenizer itr = new StringTokenizer(value.toString(), "\n");
            while (itr.hasMoreTokens()) {
                String csvLine = itr.nextToken();
                if (csvLine.startsWith("Date")) {
                    continue;
                }
                String[] csvVals = csvLine.split(",");
                country.set(csvVals[1]);
                double val = csvVals.length < 3 ? 0.0 : Double.parseDouble(csvVals[2]);

                if (val > 0) {
                    currencyPrice.set(val);
                    context.write(country, currencyPrice);
                }
            }
        }
    }

    public static class AvgReducer
            extends Reducer<Text,DoubleWritable,Text,DoubleWritable> {
        private final DoubleWritable result = new DoubleWritable();

        public void reduce(Text key, Iterable<DoubleWritable> values,
                           Context context
        ) throws IOException, InterruptedException {
            double sum = 0.0;
            int count = 0;
            for (DoubleWritable val : values) {
                count++;
                sum += val.get();
            }
            result.set(sum/count);
            context.write(key, result);
        }
    }

    public static void main(String[] args) throws Exception {
        Configuration conf = new Configuration();
        Job job = Job.getInstance(conf, "currency average");
        job.setJarByClass(CurrencyAvg.class);
        job.setMapperClass(CurrencyAvg.TokenizerMapper.class);
        job.setCombinerClass(CurrencyAvg.AvgReducer.class);
        job.setReducerClass(CurrencyAvg.AvgReducer.class);
        job.setOutputKeyClass(Text.class);
        job.setOutputValueClass(DoubleWritable.class);
        //FileInputFormat.addInputPath(job, new Path(args[0]));
        NLineInputFormat.addInputPath(job, new Path(args[0]));
        NLineInputFormat.setNumLinesPerSplit(job, 1000);
        FileOutputFormat.setOutputPath(job, new Path(args[1]));
        Instant start = Instant.now();
        boolean succ = job.waitForCompletion(true);
        Instant end = Instant.now();
        System.out.println(Duration.between(start, end));
        System.exit(succ ? 0 : 1);
    }
}
